<?php

namespace FacturaPyme;

use Exception;
use DateTimeImmutable;

defined('FACTURAPYME_VER') || exit;

class Pedido
{
    public $order;
    public $id;
    public $estado = null;
    public $pdf;
    public $tipo;
    public $folio;

    public function __construct($order)
    {
        $this->order = $order;
        $this->id = $order->get_id();
        $this->load();
    }

    public function load()
    {
        $row = db()->get_row('SELECT tipo_dte, folio, estado, pdf FROM ' . tableName('order_dte')
            . " WHERE order_id ='" . $this->id . "'");
        if (!$row) {
            return;
        }
        $this->estado = $row->estado;
        $this->pdf = $row->pdf;
        $this->tipo = $row->tipo_dte;
        $this->folio = $row->folio;
    }

    public static function fromId($id)
    {
        $order = wc_get_order($id);
        return new self($order);
    }

    public function facturarFromCheckout()
    {
        $pago = $this->order->get_payment_method();
        $configFormasPagos = get_option('facturapyme_formas_pago', []);
        if (array_get($configFormasPagos, $pago, 'M') === 'M') {
            return;
        }
        return $this->facturar();
    }

    public function getMeta()
    {
        //@TODO leer campos ingresados por usuario en Checkout o en el Backend
        //Esto es para las Facturas.
        $data = [];
        $fields = [
            'facturapyme_rut_dte' => 'facturapyme_rut_field',
            'facturapyme_razon_social_dte' => 'billing_company',
            'facturapyme_giro_dte' => 'facturapyme_giro_field',
            'facturapyme_comuna_dte' => 'facturapyme_comuna_field',
        ];
        foreach ($fields as $index => $default) {
            $data[$index] = $this->order->get_meta($index);
            if (!$data[$index]) {
                $option = get_option($index);
                $option = $option ? $option : $default;
                $function = 'get_'.$option;
                if (is_callable(array($this->order, $function )) ) {
                    $data[$index] = $this->order->{$function}();
                } else {
                    $data[$index] = $this->order->get_meta($option);
                }
            }
        }
        $tipo = $this->order->get_meta('tipo_dte');
        if ($tipo) {
            $data['tipo_dte'] = $tipo;
        }
        $data['direccion'] = $this->order->get_billing_address_1();
        $dir2 = $this->order->get_billing_address_2();
        if ($dir2) {
            $data['direccion'] .= ', '.$dir2;
        }
        return $data;
    }
    public function setMeta($metaData)
    {
        $fields = [
            'facturapyme_rut_dte',
            'facturapyme_razon_social_dte',
            'facturapyme_giro_dte',
            'facturapyme_comuna_dte',
            'tipo_dte',
        ];
        foreach ($fields as $key) {
            if (!isset($metaData[$key])) {
                continue;
            }
            $value = $metaData[$key];
            update_post_meta($this->id, $key, $value);
        }
    }
    public function transformData($data)
    {
        if (!$data) {
            return $data;
        }
        $receptor = [
            'rut' => '66666666-6',
            'razon_social' => 'Publico General',
            'giro' => 'Publico General',
            'direccion' => 'Publico General',
            'comuna' => 'La Florida',
        ];
        $rut = cleanRut(array_get($data, 'facturapyme_rut_dte', ''));
        if (!validaRut($rut)) {
            $data['receptor'] = $receptor;
            return $data;
        }
        $receptor['rut'] = $rut;
        $fields = [
            'facturapyme_razon_social_dte',
            'facturapyme_giro_dte',
            'facturapyme_comuna_dte'
        ];
        foreach($fields as $field) {
            $cleanField = str_replace(['facturapyme_', '_dte'], '', $field);
            $receptor[$cleanField] = sanitize_text_field(array_get($data, $field, $receptor[$cleanField]));
        }
        if ($data['direccion']) {
            $receptor['direccion'] = $data['direccion'];
        }
        $data['receptor'] = $receptor;
        return $data;
    }


    public function facturar()
    {
        if (!$this->sePuedeFacturar() || facturado($this->id)) {
            return false;
        }
        $data = $this->transformData($this->getMeta());
        $facturapyme = get_option('facturapyme_api_connection');
        $url = array_get($facturapyme, 'url');
        $key = array_get($facturapyme, 'key');
        throw_if(!$url || !$key, new Exception('Api No iniciada'));
        if (!isset($data['tipo_dte'])) {
            $data['tipo_dte'] = get_option('facturapyme_default_dte', 39);
        }
        $this->tipo_dte = $data['tipo_dte'];
        if (is_null($this->estado)) {
            $this->estado = 0;
            $insertData = [
                'order_id' => $this->id,
                'tipo_dte' => $this->tipo_dte,
                'estado' => $this->estado,
            ];
            db()->insert(tableName('order_dte'), $insertData);
        }
        $documento = $this->preparaDocumento($data);
        $response = '';
        $status = 1;
        $logId = null;
        $logTable = tableName('order_dte_log');
        try {
            db()->insert($logTable, [
                'order_id' => $this->id,
                'fecha' => current_time('Y-m-d H:i:s'),
                'solicitud' => json_encode($documento, JSON_PRETTY_PRINT),
                'estado' => 0,
            ]);
            $logId = db()->insert_id;
            $sdk = new SDK($url, $key);
            $sdk->enviaDTE($documento);
            $response = 'OK!';
        } catch (Exception $e) {
            $status = $e->getCode() > 400 ? 2 : 3;
            $response = $e->getMessage();
            db()->update(
                tableName('order_dte'),
                ['estado' => 2],
                ['order_id' => $this->id]
            );
        }
        throw_if(!$logId, new Exception('Sin LogId'));
        db()->query('DELETE FROM ' . $logTable . "  WHERE order_id = '" . $this->id . "' AND id <> '" . $logId . "'");
        db()->update(
            $logTable,
            ['respuesta' => $response, 'estado' => $status, ],
            ['id' => $logId]
        );
    }

    public function failedStatus()
    {
        return in_array($this->order->get_status(), ['pending', 'failed']);
    }

    public function sePuedeFacturar()
    {
        if (is_null($this->estado)) {
            return !$this->failedStatus();
        }
        if ($this->estado == 1 || $this->estado == 0) {
            return false;
        }
        $lastMinute = (new DateTimeImmutable('-1 minute', wp_timezone()))->format('Y-m-d H:i:s');
        $count = db()->get_var('SELECT COUNT(*) FROM ' . tableName('order_dte_log')
            . " WHERE order_id = '" . $this->id
            . "' AND estado = 0 AND fecha >= '" . $lastMinute . "'");
        return $count == 0;
    }

    public function getDocumento()
    {
        $dtes = [
            33 => 'Factura Electrónica',
            34 => 'Factura Exenta Electrónica',
            39 => 'Boleta Electrónica',
            41 => 'Boleta Exenta Electrónica',
        ];
        $documento = array_get($dtes, $this->tipo, $this->tipo) . ' ' . nFormat($this->folio);
        if ($this->pdf) {
            $documento = '<a href="' . $this->pdf . '" target="_blank">' . $documento . '</a>';
        }
        return $documento;
    }

    protected function preparaDocumento($data)
    {
        $order = $this->order;
        $refFolio = $order->get_order_number();
        $factor = 1.19;
        if ($data['tipo_dte'] == 34 || $data['tipo_dte'] == 41) {
            //Documentos Exentos...
            $factor = 1;
        }
        if ($data['tipo_dte'] == 33 || $data['tipo_dte'] == 34) {
            //Datos del Receptor...
            throw_if($data['receptor']['rut'] === '66666666-6', new Exception('No se Puede facturar a este rut'));
        }
        $bodega = DataSource::bodega();
        $documento = [
            'id' => $this->id,
            'src' => 'woocommerce',
            'tipo_dte' => $data['tipo_dte'],
            'fecha_emision' => current_time('Y-m-d'),
            'receptor' => $data['receptor'],
            'detalle' => $this->detalleDocumento($order, $factor, $bodega),
            'referencias' => [
                [
                    'tipo_dte' => 801,
                    'folio' => $refFolio ? $refFolio : $this->id,
                    'fecha' => substr($order->get_date_created(), 0, 10),
                ]
            ]
        ];
        if ($bodega) {
            $documento['bodega_id'] = $bodega;
        }

        return $documento;
    }
    protected function detalleDocumento($order, $factor, $bodega)
    {
        $detalleDoc = [];
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            $detalle = [
                'cantidad' => $item->get_quantity(),
                'nombre' => $item->get_name(),
                'barcode' => $product->get_sku()
            ];
            $detalle['precio_unitario'] = round($item->get_total() / $detalle['cantidad'] / $factor, 4);
            if ($bodega) {
                $detalle['id'] = $item->get_product_id();
            }
            $detalleDoc[] = $detalle;
        }
        $gastosEnvio = $order->get_shipping_total();
        if ($gastosEnvio > 0 && get_option('facturapyme_incluye_envio', 'S') === 'S') {
            $detalle = [
                'nombre' => 'Gastos de Envío',
                'precio_unitario' => round($gastosEnvio / $factor, 4),
                'cantidad' => 1
            ];
            $detalleDoc[] = $detalle;
        }
        return $detalleDoc;
    }
}
/*
order_dte
estado
0 --> Esperando respuesta
1 --> Ok
2 --> Error

order_dte_log
0 -> Iniciando.
1 --> Ok
2 --> Error Api
3 --> Error Interno

*/
